import numpy as np
import tensorflow as tf
from tensorflow.python.ops import tensor_array_ops, control_flow_ops


class Generator(object):
    def __init__(self, num_vocabulary, batch_size, hidden_dim,
                 sequence_length, start_token, discriminator=None, sess=None,
                 learning_rate=0.001, reward_gamma=0.95):
        self.num_vocabulary = num_vocabulary
        self.batch_size = batch_size
        self.emb_dim = num_vocabulary
        self.hidden_dim = hidden_dim
        self.sequence_length = sequence_length
        self.start_token = tf.constant([start_token] * self.batch_size, dtype=tf.int32)
        self.learning_rate = tf.Variable(float(learning_rate), trainable=False)
        self.reward_gamma = reward_gamma
        self.discriminator = discriminator
        self.g_params = []
        self.d_params = []
        self.grad_clip = 5.0
        self.tau = 10
        self.sess = sess

        self.expected_reward = tf.Variable(tf.zeros([self.sequence_length]))

        with tf.variable_scope('generator'):
            self.g_recurrent_unit = self.create_recurrent_unit(self.g_params)  # maps h_tm1 to h_t for generator
            self.g_output_unit = self.create_output_unit(self.g_params)  # maps h_t to o_t (output token logits)

        # placeholder definition
        self.x = tf.placeholder(tf.int32, shape=[self.batch_size,
                                                 self.sequence_length])  # sequence of tokens generated by generator

        onehot = tf.constant(np.eye(num_vocabulary), dtype=tf.float32)
        # processed for batch
        with tf.device("/cpu:0"):
            self.processed_x = tf.transpose(tf.nn.embedding_lookup(onehot, self.x),
                                            perm=[1, 0, 2])  # seq_length x batch_size x emb_dim

        # Initial states
        self.h_0 = tf.placeholder(tf.float32, shape=[batch_size, hidden_dim])
        self.c_0 = tf.placeholder(tf.float32, shape=[batch_size, hidden_dim])
        self.h0 = tf.stack([self.h_0, self.c_0])

        gen_o = tensor_array_ops.TensorArray(dtype=tf.float32, size=self.sequence_length,
                                             dynamic_size=False, infer_shape=True)
        gen_x = tensor_array_ops.TensorArray(dtype=tf.int32, size=self.sequence_length,
                                             dynamic_size=False, infer_shape=True)
        gen_ot = tensor_array_ops.TensorArray(dtype=tf.float32, size=self.sequence_length,
                                               dynamic_size=False, infer_shape=True)

        def _g_recurrence(i, x_t, h_tm1, gen_o, gen_x, gen_ot):
            h_t = self.g_recurrent_unit(x_t, h_tm1)  # hidden_memory_tuple
            h_t = self.add_gumbel(h_t)
            o_t = self.g_output_unit(h_t)  # batch x vocab , logits not prob
            x_tp1 = tf.nn.softmax(o_t / self.tau)
            next_token = tf.cast(tf.argmax(o_t, axis=1), tf.int32)
            gen_o = gen_o.write(i, tf.reduce_sum(tf.multiply(tf.one_hot(next_token, self.num_vocabulary, 1.0, 0.0),
                                                             tf.nn.softmax(o_t)), 1))  # [batch_size] , prob
            gen_x = gen_x.write(i, next_token)  # indices, batch_size
            gen_ot = gen_ot.write(i, x_tp1)
            return i + 1, x_tp1, h_t, gen_o, gen_x, gen_ot

        _, _, h_t, self.gen_o, self.gen_x, self.gen_ot = control_flow_ops.while_loop(
            cond=lambda i, _1, _2, _3, _4, _5: i < self.sequence_length,
            body=_g_recurrence,
            loop_vars=(tf.constant(0, dtype=tf.int32),
                       tf.nn.embedding_lookup(onehot, self.start_token), self.h0, gen_o, gen_x, gen_ot))

        self.gen_x = self.gen_x.stack()  # seq_length x batch_size
        self.gen_x = tf.transpose(self.gen_x, perm=[1, 0])  # batch_size x seq_length

        self.gen_ot = self.gen_ot.stack()
        self.gen_ot = tf.transpose(self.gen_ot, perm=[1, 0, 2]) #  batch_size x seq_length x num_vocabulary

        # todo
        self.h_t_ = h_t

        # supervised pretraining for generator
        g_predictions = tensor_array_ops.TensorArray(
            dtype=tf.float32, size=self.sequence_length,
            dynamic_size=False, infer_shape=True)


        def _pretrain_recurrence(i, x_t, h_tm1, g_predictions):
            h_t = self.g_recurrent_unit(x_t, h_tm1)
            h_t = self.add_gumbel(h_t)
            o_t = self.g_output_unit(h_t)
            g_predictions = g_predictions.write(i, tf.nn.softmax(o_t))  # batch x vocab_size
            x_tp1 = tf.nn.softmax(o_t / self.tau)
            return i + 1, x_tp1, h_t, g_predictions

        _, _, _, self.g_predictions = control_flow_ops.while_loop(
            cond=lambda i, _1, _2, _3: i < self.sequence_length,
            body=_pretrain_recurrence,
            loop_vars=(tf.constant(0, dtype=tf.int32),
                       tf.nn.embedding_lookup(onehot, self.start_token),
                       self.h0, g_predictions))

        self.g_predictions = tf.transpose(self.g_predictions.stack(),
                                          perm=[1, 0, 2])  # batch_size x seq_length x vocab_size

        # pretraining loss
        self.pretrain_loss = -tf.reduce_sum(
            tf.one_hot(tf.to_int32(tf.reshape(self.x, [-1])), self.num_vocabulary, 1.0, 0.0) * tf.log(
                tf.clip_by_value(tf.reshape(self.g_predictions, [-1, self.num_vocabulary]), 1e-20, 1.0)
            )
        ) / (self.sequence_length * self.batch_size)

        # training updates
        pretrain_opt = self.g_optimizer(self.learning_rate)

        self.pretrain_grad, _ = tf.clip_by_global_norm(tf.gradients(self.pretrain_loss, self.g_params), self.grad_clip)
        self.pretrain_updates = pretrain_opt.apply_gradients(zip(self.pretrain_grad, self.g_params))

        #######################################################################################################
        #  Unsupervised Training
        #######################################################################################################

        d_pred = self.discriminator.predict(self.gen_ot)
        neg_score = tf.slice(tf.reshape(d_pred, shape=[2,batch_size]), [0, 0], [1, batch_size])
        self.g_loss = tf.negative(tf.divide(tf.divide(neg_score, tf.subtract(1.0, neg_score)), batch_size))
        g_opt = self.g_optimizer(self.learning_rate)

        g_grad, _ = tf.clip_by_global_norm(tf.gradients(self.g_loss, self.g_params), self.grad_clip)
        self.g_updates = g_opt.apply_gradients(zip(g_grad, self.g_params))
        pass


    def add_gumbel(self, hidden_memory_tuple):
        hidden_state, c_prev = tf.unstack(hidden_memory_tuple)
        hidden_state = tf.add(hidden_state, self.get_gumbel(tf.shape(hidden_state)))
        return tf.stack([hidden_state, c_prev])

    def get_gumbel(self, shape, eps=1e-3):
        """Sample from Gumbel(0, 1)"""
        u = tf.random_uniform(shape, minval=0, maxval=1, dtype=tf.float32)
        return -tf.log(-tf.log(u + eps) + eps)

    def generate(self, sess):
        z_h0 = np.random.uniform(low=-1, high=1, size=[self.batch_size, self.hidden_dim])
        z_c0 = np.random.uniform(low=-1, high=1, size=[self.batch_size, self.hidden_dim])
        feed = {
            self.h_0: z_h0,
            self.c_0: z_c0,
        }
        outputs = sess.run(self.gen_x, feed)
        return outputs

    def get_nll(self, sess, batch):
        z_h0 = np.random.uniform(low=-1, high=1, size=[self.batch_size, self.hidden_dim])
        z_c0 = np.random.uniform(low=-1, high=1, size=[self.batch_size, self.hidden_dim])
        feed = {
            self.h_0: z_h0,
            self.c_0: z_c0,
            self.x: batch
        }
        return sess.run(self.pretrain_loss, feed)

    def pretrain_step(self, sess, x):
        z_h0 = np.random.uniform(low=-1, high=1, size=[self.batch_size, self.hidden_dim])
        z_c0 = np.random.uniform(low=-1, high=1, size=[self.batch_size, self.hidden_dim])
        feed = {
            self.h_0: z_h0,
            self.c_0: z_c0,
            self.x: x
        }
        outputs = sess.run([self.pretrain_updates, self.pretrain_loss], feed_dict=feed)
        h_t = sess.run([self.h_t_], feed_dict=feed)
        return outputs

    def unsupervised_train(self, sess):
        z_h0 = np.random.uniform(low=-1, high=1, size=[self.batch_size, self.hidden_dim])
        z_c0 = np.random.uniform(low=-1, high=1, size=[self.batch_size, self.hidden_dim])
        feed = {
            self.h_0: z_h0,
            self.c_0: z_c0,
        }
        outputs, _ = sess.run([self.g_loss, self.g_updates], feed_dict=feed)
        pass

    def init_matrix(self, shape):
        return tf.random_normal(shape, stddev=0.1)

    def init_vector(self, shape):
        return tf.zeros(shape)

    def create_recurrent_unit(self, params):
        # Weights and Bias for input and hidden tensor
        self.Wi = tf.Variable(self.init_matrix([self.emb_dim, self.hidden_dim]))
        self.Ui = tf.Variable(self.init_matrix([self.hidden_dim, self.hidden_dim]))
        self.bi = tf.Variable(self.init_matrix([self.hidden_dim]))

        self.Wf = tf.Variable(self.init_matrix([self.emb_dim, self.hidden_dim]))
        self.Uf = tf.Variable(self.init_matrix([self.hidden_dim, self.hidden_dim]))
        self.bf = tf.Variable(self.init_matrix([self.hidden_dim]))

        self.Wog = tf.Variable(self.init_matrix([self.emb_dim, self.hidden_dim]))
        self.Uog = tf.Variable(self.init_matrix([self.hidden_dim, self.hidden_dim]))
        self.bog = tf.Variable(self.init_matrix([self.hidden_dim]))

        self.Wc = tf.Variable(self.init_matrix([self.emb_dim, self.hidden_dim]))
        self.Uc = tf.Variable(self.init_matrix([self.hidden_dim, self.hidden_dim]))
        self.bc = tf.Variable(self.init_matrix([self.hidden_dim]))
        params.extend([
            self.Wi, self.Ui, self.bi,
            self.Wf, self.Uf, self.bf,
            self.Wog, self.Uog, self.bog,
            self.Wc, self.Uc, self.bc])

        def unit(x, hidden_memory_tm1):
            previous_hidden_state, c_prev = tf.unstack(hidden_memory_tm1)

            # Input Gate
            i = tf.sigmoid(
                tf.matmul(x, self.Wi) +
                tf.matmul(previous_hidden_state, self.Ui) + self.bi
            )

            # Forget Gate
            f = tf.sigmoid(
                tf.matmul(x, self.Wf) +
                tf.matmul(previous_hidden_state, self.Uf) + self.bf
            )

            # Output Gate
            o = tf.sigmoid(
                tf.matmul(x, self.Wog) +
                tf.matmul(previous_hidden_state, self.Uog) + self.bog
            )

            # New Memory Cell
            c_ = tf.nn.tanh(
                tf.matmul(x, self.Wc) +
                tf.matmul(previous_hidden_state, self.Uc) + self.bc
            )

            # Final Memory cell
            c = f * c_prev + i * c_

            # Current Hidden state
            current_hidden_state = o * tf.nn.tanh(c)

            return tf.stack([current_hidden_state, c])

        return unit

    def create_output_unit(self, params):
        self.Wo = tf.Variable(self.init_matrix([self.hidden_dim, self.num_vocabulary]))
        self.bo = tf.Variable(self.init_matrix([self.num_vocabulary]))
        params.extend([self.Wo, self.bo])

        def unit(hidden_memory_tuple):
            hidden_state, c_prev = tf.unstack(hidden_memory_tuple)
            logits = tf.matmul(hidden_state, self.Wo) + self.bo
            return logits

        return unit

    def g_optimizer(self, *args, **kwargs):
        return tf.train.AdamOptimizer(*args, **kwargs)

    def set_similarity(self, valid_examples=None, pca=True):
        one_hot = tf.constant(np.eye(self.num_vocabulary))
        if valid_examples == None:
            if pca:
                valid_examples = np.array(range(20))
            else:
                valid_examples = np.array(range(self.num_vocabulary))
        self.valid_dataset = tf.constant(valid_examples, dtype=tf.int32)
        self.norm = tf.sqrt(tf.reduce_sum(tf.square(one_hot), 1, keep_dims=True))
        self.normalized_embeddings = one_hot / self.norm
        # PCA
        if self.num_vocabulary >= 20 and pca == True:
            emb = tf.matmul(self.normalized_embeddings, tf.transpose(self.normalized_embeddings))
            s, u, v = tf.svd(emb)
            u_r = tf.strided_slice(u, begin=[0, 0], end=[20, self.num_vocabulary], strides=[1, 1])
            self.normalized_embeddings = tf.matmul(u_r, self.normalized_embeddings)
        self.valid_embeddings = tf.nn.embedding_lookup(
            self.normalized_embeddings, self.valid_dataset)
        self.similarity = tf.matmul(self.valid_embeddings, tf.transpose(self.normalized_embeddings))
